<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

/**
 * Cron Class
 *
 * @package Wojo Framework
 * @author wojoscripts.com
 * @copyright 2023
 * @version 5.00: Cron.php, v1.00 7/19/2023 8:06 AM Gewa Exp $
 *
 */

use Stripe\Exception\ApiErrorException;
use Stripe\Exception\CardException;
use Stripe\PaymentIntent;
use Stripe\Stripe;

if (!defined('_WOJO')) {
    die('Direct access to this location is not allowed.');
}


if (!class_exists('Cron')) {

class Cron
{
    /**
     * run
     *
     * @param int $days
     * @return void
     * @throws ApiErrorException
     * @throws NotFoundException
     * @throws \PHPMailer\PHPMailer\Exception
     */
    public static function run(int $days): void
    {
        $data = self::expireMemberships($days); // Memorizza i dati degli utenti con abbonamenti scaduti
        self::runStripe(); // Esegue i pagamenti tramite Stripe
        self::sendEmails($data); // Invia email agli utenti
    }

    /**
     * expireMemberships
     *
     * @param int $days
     * @return mixed
     */
    public static function expireMemberships(int $days): mixed
    {
        // Seleziona gli utenti con abbonamenti in scadenza
        $sql = "
        SELECT u.id, CONCAT(u.fname,' ',u.lname) as fullname, u.email, u.mem_expire, m.id AS mid
          FROM " . User::mTable . ' as u
          LEFT JOIN ' . Membership::mTable . " AS m ON m.id = u.membership_id
          WHERE u.active = ?
          AND u.membership_id <> 0
          AND u.mem_expire <= DATE_ADD(DATE(NOW()), INTERVAL $days DAY)
        ";
        
        $result = Database::Go()->rawQuery($sql, array('y'))->run();
        
        if ($result) {
            $query = 'UPDATE ' . User::mTable . ' SET mem_expire = NULL, membership_id = CASE ';
            $id_list = '';
            foreach ($result as $usr) {
                $query .= ' WHEN id = ' . $usr->id . ' THEN membership_id = 0';
                $id_list .= $usr->id . ',';
            }
            $id_list = rtrim($id_list, ',');
            $query .= '
                END
                WHERE id IN (' . $id_list . ')';
            Database::Go()->rawQuery($query)->run();
        }
        
        return $result;
    }

    /**
     * runStripe
     *
     * @return void
     * @throws NotFoundException
     * @throws ApiErrorException
     */
    public static function runStripe(): void
    {
        // Seleziona gli utenti da rinnovare
        $sql = "
        SELECT u.id AS uid, CONCAT(u.fname,' ',u.lname) as fullname, u.email, cj.amount, cj.id as cid, cj.membership_id, cj.stripe_customer, cj.stripe_pm
          FROM " . Core::cjTable . ' as cj
          LEFT JOIN ' . User::mTable . ' as u ON u.id = cj.user_id
          WHERE u.active = ?
          AND DATE(cj.renewal) = CURDATE()
        ';
        
        $data = Database::Go()->rawQuery($sql, array('y'))->run();
        
        // Configura Stripe
        require_once BASEPATH . '/gateways/stripe/vendor/autoload.php';
        $key = Database::Go()->select(Core::gTable)->where('name', 'stripe', '=')->first()->run();
        Stripe::setApiKey($key->extra);
        
        if ($data) {
            try {
                foreach ($data as $row) {
                    // Crea un PaymentIntent su Stripe
                    PaymentIntent::create([
                        'amount' => round($row->amount * 100),
                        'currency' => $key->extra2,
                        'customer' => $row->stripe_customer,
                        'payment_method' => $row->stripe_pm,
                        'off_session' => true,
                        'confirm' => true,
                    ]);

                    // Dati per la transazione
                    $data = array(
                        'txn_id' => time(),
                        'membership_id' => $row->membership_id, // Associa correttamente l'ID dell'abbonamento
                        'user_id' => $row->uid,
                        'rate_amount' => $row->amount, // Cambiato 'price' in 'amount'
                        'total' => $row->amount,
                        'currency' => $key->extra2,
                        'pp' => 'Stripe',
                        'status' => 1,
                    );
                    
                    // Inserisci la transazione nel database
                    $last_id = Database::Go()->insert(Membership::pTable, $data)->run();
                    
                    // Dati per l'aggiornamento dell'abbonamento dell'utente
                    $udata = array(
                        'transaction_id' => $last_id,
                        'user_id' => $row->uid,
                            'membership_id' => $row->membership_id,  // Aggiungi il membership_id

                        'expire' => Membership::calculateDays($row->membership_id), // Calcola la scadenza dell'abbonamento
                        'recurring' => 1,
                        'active' => 1,
                    );
                    
                    // Inserisci nell'utente
                    Database::Go()->insert(Membership::umTable, $udata)->run();
                    
                    // Aggiorna il record dell'utente con l'abbonamento attivo
                    Database::Go()->update('users', array('mem_expire' => $udata['expire'], 'membership_id' => $row->membership_id))->where('id', $row->uid, '=')->run();
                    
                    // Aggiorna il record cron
                    Database::Go()->update(Core::cjTable, array('renewal' => $udata['expire']))->where('id', $row->cid, '=')->run();
                }
                
            } catch (CardException $e) {
                // Gestisci eccezione Stripe
            }
        }
    }

    /**
 * sendEmails
 *
 * @param mixed $data
 * @return void
 * @throws \PHPMailer\PHPMailer\Exception
 */
public static function sendEmails(mixed $data): void
{
    if ($data) {
        $core = App::Core();

        // Costruisce il corpo della mail con l'elenco degli utenti che hanno rinnovato
        $body = "I seguenti account hanno rinnovato l'abbonamento:\n\n";
        foreach ($data as $row) {
            $body .= "Nome: {$row->fullname}, Email: {$row->email}\n";
        }

        $mailer = Mailer::sendMail();
        $mailer->Subject = "Rinnovi Abbonamenti";
        $mailer->setFrom($core->site_email, $core->company);
        $mailer->isHTML(false); // Testo semplice

        $mailer->Body = $body;
        $mailer->addAddress($core->site_email, "Admin");

        try {
            $mailer->send();
        } catch (Exception $e) {
            error_log("Errore nell'invio dell'email di notifica admin: " . $e->getMessage());
        }
    }
}

}
    
}